import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  UseGuards,
  Req,
  SetMetadata,
  HttpException,
  HttpStatus,
} from '@nestjs/common';
import { UsersService } from './users.service';
import { CreateUserDto } from './dto/create-user.dto';
import { UpdateUserDto } from './dto/update-user.dto';
import { PrismaService } from '../prisma/prisma.service';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';
import { ApiBearerAuth } from '@nestjs/swagger';
import { AuthGuard } from '../auth/guards/auth.guard';

@Controller('users')
export class UsersController {
  constructor(
    private readonly usersService: UsersService,
    private prisma: PrismaService,
  ) {}

  @ApiBearerAuth()
  @SetMetadata('permissions', ['create_user'])
  @UseGuards(JwtAuthGuard, AuthGuard)
  @Post()
  create(@Body() createUserDto: CreateUserDto) {
    return this.usersService.create(createUserDto);
  }

  @ApiBearerAuth()
  @SetMetadata('permissions', ['view_user'])
  @UseGuards(JwtAuthGuard, AuthGuard)
  @Get()
  findAll(@Req() req: any) {
    const page = req.query.page ? req.query.page : '1';
    return this.usersService.findAll(
      page,
      `${req.protocol}://${req.get('Host')}${req.url}`,
    );
  }

  @ApiBearerAuth()
  @SetMetadata('permissions', ['view_user'])
  @UseGuards(JwtAuthGuard, AuthGuard)
  @Get('/roles')
  roles() {
    return this.usersService.roles();
  }

  @ApiBearerAuth()
  @SetMetadata('permissions', ['view_user'])
  @UseGuards(JwtAuthGuard, AuthGuard)
  @Get('/list')
  all() {
    return this.usersService.users();
  }

  @ApiBearerAuth()
  @UseGuards(JwtAuthGuard)
  @Get('me')
  me(@Req() req: any) {
    return this.usersService.me(req.user.id);
  }

  @ApiBearerAuth()
  @SetMetadata('permissions', ['view_user'])
  @UseGuards(JwtAuthGuard, AuthGuard)
  @Get(':id')
  findOne(@Param('id') id: string) {
    return this.usersService.findOne(id).catch((e) => {
      throw new HttpException('Not Found!', HttpStatus.NOT_FOUND);
    });
  }

  @ApiBearerAuth()
  @SetMetadata('permissions', ['edit_user'])
  @UseGuards(JwtAuthGuard, AuthGuard)
  @Patch(':id')
  update(@Param('id') id: string, @Body() updateUserDto: UpdateUserDto) {
    return this.usersService.update(id, updateUserDto);
  }

  @ApiBearerAuth()
  @SetMetadata('permissions', ['destroy_user'])
  @UseGuards(JwtAuthGuard, AuthGuard)
  @Delete(':id')
  remove(@Param('id') id: string) {
    return this.usersService.remove(id);
  }
}
