import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { CreateNotificationDto } from './dto/create-notification.dto';
import { UpdateNotificationDto } from './dto/update-notification.dto';
import { PrismaService } from '../prisma/prisma.service';

@Injectable()
export class NotificationsService {
  constructor(private prisma: PrismaService) {}

  create(createNotificationDto: CreateNotificationDto, user_id) {
    const receivers = JSON.parse(createNotificationDto.receiver_id);
    return this.prisma.notification.create({
      data: {
        author_id: user_id,
        file_id: createNotificationDto.file_id,
        title: createNotificationDto.title,
        content: createNotificationDto.content,
        Notification_Receiver: {
          createMany: {
            data: receivers.map((el) => {
              return {
                user_id: el,
              };
            }),
          },
        },
      },
    });
  }

  findAll() {
    return this.prisma.notification.findMany({
      select: {
        id: true,
        title: true,
        author: {
          select: {
            userDetail: {
              select: {
                first_name: true,
                last_name: true,
              },
            },
          },
        },
        Notification_Receiver: {
          select: {
            user: {
              select: {
                userDetail: {
                  select: {
                    first_name: true,
                    last_name: true,
                  },
                },
              },
            },
          },
        },
        createdAt: true,
      },
    });
  }

  findOne(id: string) {
    return this.prisma.notification.findUnique({
      where: { id },
      select: {
        id: true,
        title: true,
        content: true,
        createdAt: true,
        author: {
          select: {
            userDetail: {
              select: {
                first_name: true,
                last_name: true,
              },
            },
          },
        },
        Notification_Receiver: {
          select: {
            isRead: true,
            user: {
              select: {
                userDetail: {
                  select: {
                    first_name: true,
                    last_name: true,
                  },
                },
              },
            },
          },
        },
      },
    });
  }

  async update(
    id: string,
    updateNotificationDto: UpdateNotificationDto,
    user_id,
  ) {
    const notification = await this.prisma.notification.findUnique({
      where: { id },
      select: {
        id: true,
        Notification_Receiver: {
          select: {
            user_id: true
          }
        }
      },
    });

    if (notification.Notification_Receiver.some(x=>x.user_id === user_id)) {
      await this.prisma.notification_Receiver.updateMany({
        where: {
          AND: {
            user_id: user_id,
            notification_id: notification.id,
          },
        },
        data: {
          isRead: updateNotificationDto.isRead === 'true',
        },
      });
    } else {
      throw new HttpException("You can't", HttpStatus.FORBIDDEN);
    }
    return true;
  }

  async remove(id: string) {
    await this.prisma.notification_Receiver.deleteMany({
      where: {
        notification_id: id,
      },
    });
    return this.prisma.notification.delete({
      where: { id },
    });
  }

  users() {
    return this.prisma.user.findMany({
      select: {
        id: true,
        userDetail: {
          select: {
            first_name: true,
            last_name: true,
          },
        },
      },
    });
  }

  myUnreadNotifications(id: string) {
    return this.prisma.notification.findMany({
      where: {
        Notification_Receiver: {
          every: {
            isRead: false,
            user_id: id,
          },
        },
      },
      select: {
        id: true,
        title: true,
        content: true,
        createdAt: true,
        Notification_Receiver: {
          select: {
            isRead: true,
          },
        },
      },
      take: 5,
    });
  }

  myNotifications(id) {
    return this.prisma.notification.findMany({
      where: {
        Notification_Receiver: {
          every: {
            user_id: id,
          },
        },
      },
    });
  }
}
