import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  Req,
  SetMetadata,
  UseGuards,
} from '@nestjs/common';
import { NotificationsService } from './notifications.service';
import { CreateNotificationDto } from './dto/create-notification.dto';
import { UpdateNotificationDto } from './dto/update-notification.dto';
import { ApiBearerAuth } from '@nestjs/swagger';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';
import { AuthGuard } from '../auth/guards/auth.guard';

@Controller('notifications')
export class NotificationsController {
  constructor(private readonly notificationsService: NotificationsService) {}

  @Post()
  @ApiBearerAuth()
  @SetMetadata('permissions', ['create_notification'])
  @UseGuards(JwtAuthGuard, AuthGuard)
  create(
    @Body() createNotificationDto: CreateNotificationDto,
    @Req() req: any,
  ) {
    return this.notificationsService.create(createNotificationDto, req.user.id);
  }

  @Get()
  @ApiBearerAuth()
  @SetMetadata('permissions', ['view_notification'])
  @UseGuards(JwtAuthGuard, AuthGuard)
  findAll() {
    return this.notificationsService.findAll();
  }

  @Get('users')
  @ApiBearerAuth()
  @SetMetadata('permissions', ['view_notification'])
  @UseGuards(JwtAuthGuard, AuthGuard)
  users() {
    return this.notificationsService.users();
  }

  @Get('my')
  @ApiBearerAuth()
  @UseGuards(JwtAuthGuard)
  mynotifs(@Req() req: any) {
    return this.notificationsService.myUnreadNotifications(req.user.id);
  }

  @Get('my-list')
  @ApiBearerAuth()
  @UseGuards(JwtAuthGuard)
  mynotifslist(@Req() req: any) {
    return this.notificationsService.myNotifications(req.user.id);
  }

  @Get(':id')
  @ApiBearerAuth()
  @SetMetadata('permissions', ['view_notification'])
  @UseGuards(JwtAuthGuard, AuthGuard)
  findOne(@Param('id') id: string) {
    return this.notificationsService.findOne(id);
  }

  @Patch(':id')
  @ApiBearerAuth()
  @UseGuards(JwtAuthGuard)
  update(
    @Param('id') id: string,
    @Body() updateNotificationDto: UpdateNotificationDto,
    @Req() req: any,
  ) {
    return this.notificationsService.update(
      id,
      updateNotificationDto,
      req.user.id,
    );
  }

  @Delete(':id')
  @ApiBearerAuth()
  @SetMetadata('permissions', ['destroy_notification'])
  @UseGuards(JwtAuthGuard, AuthGuard)
  remove(@Param('id') id: string) {
    return this.notificationsService.remove(id);
  }
}
